<?php  

/**
 * Class to help prevent scripts from running too long or running in parallel.
 * It includes functionallity to alert via slak if necessary.
 */
class schedule_helper {

	private $pname;
	private $cta;
	private $ctr;
	private $hook;
	private $fname;

	/**
	 * Constructs the schedule helper object
	 * @param string 	$process_name		Name of the script being scheduled
	 * @param int 		$checks_to_alert	Checks before an alert is sent.
	 * @param int 		$checks_to_reset	Checks before the process is assumed to have been killed and must restart
	 * @param string 	$alert_hook			Slack hook url to send alert messages
	 */
	public function __construct($process_name, $checks_to_alert, $checks_to_reset, $alert_hook = "") {

		$this->pname = $process_name;
		$this->cta = $checks_to_alert;
		$this->ctr = $checks_to_reset;
		$this->hook = $alert_hook;
		$this->fname = __DIR__ . "/schdlr/" . $process_name . ".log";

		if (!file_exists(__DIR__ . "/schdlr")) {
		    mkdir(__DIR__ . "/schdlr/", 0755, true);
		}

	}

	/**
	 * Sends a message to the hook url
	 * @param  string $message message to send
	 */
	public function alert($message) {
		if (empty($this->hook))
			return;

		$data = array (
			"text" => $message
		);

		$ch = curl_init();
		$timeout = 5;
		curl_setopt($ch, CURLOPT_HTTPHEADER, array(
		    'Content-type: application/json'
		));
		curl_setopt($ch,CURLOPT_URL,$this->hook);
		curl_setopt($ch,CURLOPT_POSTFIELDS, json_encode($data));
		curl_setopt($ch,CURLOPT_RETURNTRANSFER,1);
		curl_setopt($ch,CURLOPT_CONNECTTIMEOUT,$timeout);
		$result = curl_exec($ch);
		curl_close($ch);

	}

	/**
	 * Resets process schedule to not running
	 */
	public function finish() {
		$data = json_decode(file_get_contents($this->fname), true);
		$data['running'] = false;
		file_put_contents($this->fname,json_encode($data));
	}

	/**
	 * Checks if the script should run or not and alerts if necesary.
	 * @return bool    True if should run, False if it should not.
	 */
	public function check() {

		$data_out = array(
			'lastrun' => date('Y-m-d h:i:s'),
			'lastruntried' => date('Y-m-d h:i:s'),
			'running' => true,
			'checks' => 0
		);

		/*
			If no file then process is not running
		 */
		if (!file_exists($this->fname)) {

			file_put_contents($this->fname,json_encode($data_out));

			return true;
		}

		$data = json_decode(file_get_contents($this->fname), true);

		/*
			Checks if the process is not currently running
		 */
		if (!$data['running']){
			file_put_contents($this->fname,json_encode($data_out));

			return true;
		}
		/*
			If process is running...
		 */
		else {
			$data['checks'] = $data['checks'] + 1;

			/*
				Fire alert if necessary
			 */
			if ($data['checks'] > $this->cta) {

				$this->alert("Process ".$this->pname." has not run for ". $data['checks'] . " times.");
				$data['lastruntried'] = date('Y-m-d h:i:s');

			}

			/*
				Reset running status and try to restart process
			 */
			if ($data['checks'] > $this->ctr) {
				$this->alert("Resseting process ".$this->pname);	
				$data_out['running'] = false;			
				file_put_contents($this->fname,json_encode($data_out));
				return false;
			}


			file_put_contents($this->fname,json_encode($data));
			return false;

		}

	}

}