<?php

if (!defined('UPDATE_NAME')) define('UPDATE_NAME',true);
if (!defined('UPDATE_PRICE')) define('UPDATE_PRICE',true);
if (!defined('UPDATE_IMAGES')) define('UPDATE_IMAGES',true);
if (!defined('UPDATE_CATEGORIES')) define('UPDATE_CATEGORIES',true);
if (!defined('UPDATE_STOCK')) define('UPDATE_STOCK',true);
if (!defined('UPDATE_SALE')) define('UPDATE_SALE',true);
if (!defined('UPDATE_STATUS')) define('UPDATE_STATUS',true);

if (!defined('CREATE_NAME')) define('CREATE_NAME',true);
if (!defined('CREATE_PRICE')) define('CREATE_PRICE',true);
if (!defined('CREATE_IMAGES')) define('CREATE_IMAGES',true);
if (!defined('CREATE_CATEGORIES')) define('CREATE_CATEGORIES',true);
if (!defined('CREATE_STOCK')) define('CREATE_STOCK',true);
if (!defined('CREATE_SALE')) define('CREATE_SALE',true);
if (!defined('CREATE_STATUS')) define('CREATE_STATUS',true);

/**
 * Product class to update products based on an sync db
 */
class product{
	private $options;
	private $woo;
	private $erp;
	private $category;
	private $logger;
	private $batch;
	private $batch_count;
	private $batch_size;

	/**
	 * Constructor.
	 * @param WooCommerce\Client $woo_connection The woocommerce API connection object
	 * @param erpinterface		 $erp_connection The object that implements the erp interface
	 * @param int 				 $batch_size 	 The size of batches to process
	 */
	function __construct($woo_connection, $erp_connection, $category, $batch_size = 20, $options = array()) {
		$this->woo = $woo_connection;
		$this->erp = $erp_connection;
		$this->category = $category;
		$this->options = $options;

		$params = array();
		$params['db_log'] = $this->erp;
		if (isset($_GET['do_echo'])) {
			$params['do_echo'] = true;
		}

		$this->logger = new logger('prouctos.log', $params);

		//Initialize batch
		$this->batch = array();
		$this->batch_count = 0;
		$this->batch_size = $batch_size;

	}

	/**
	 * Starts update process using traditional one-by-one approach
	 */
	public function run() {
		$products = $this->erp->get_products();

		$this->logger->i('Starting product update');

		foreach ($products as $product) {

			$this->logger->d('Processing product '.print_r($product,1));

			$product_id = $product['id_wp'];
			
			/**
			 * If product ID for woo is not set try to get it
			 */
			if ($product_id <= 0) {
				$product_id = $this->get_id($product);

				if ($product_id > 0) {

					$this->logger->d("Fetched product_id ".$product_id);

					//Update product based on ID fetched
					$this->erp->update_product_id($product['sku'], $product_id, '');

					$product['id_wp'] = $product_id;

					if (!$this->update_product($product)) {
						$this->logger->e('Unable to update product SKU: '.$product['sku']);
					} else {
						$this->logger->d('Updated product successfully!');
					}

				} else {

					//product not found, try to create
					
					$product_id = $this->create_product($product);

					$this->logger->i("Created product ".$product_id);

					if ($product_id > 0){

						$this->erp->update_product_id($product['sku'], $product_id, '');

					} else {

						$this->erp->update_product_id($product['sku'], $product_id, 'Error creating/fetching woo product');
						$this->logger->e("Error creating product SKU: ".$product['sku']);
						
					}
				}
			} else {

				/**
				 * If product had existing Woo ID
				 */
				
				if (!$this->update_product($product)) {
					$this->logger->e('Unable to update product SKU: '.$product['sku']);
				} else {
					$this->logger->d('Updated product successfully!');
				}

			}

		}

		$this->logger->i('Finished product update');

		$this->erp->close_connection();
	}


	/**
	 * Starts update process using batch approach
	 * @return [type] [description]
	 */
	public function run_batch() {
		$products = $this->erp->get_products();

		$this->logger->i('Starting product update');

		foreach ($products as $product) {

			$this->logger->d('Processing product '.print_r($product,1));

			$product_id = $product['id_wp'];
			
			/**
			 * If product ID for woo is not set try to get it
			 */
			if ($product_id <= 0) {
				$product_id = $this->get_id($product);

				if ($product_id > 0) {

					$this->logger->d("Fetched product_id ".$product_id);

					//Update product based on ID fetched
					$this->erp->update_product_id($product['sku'], $product_id, '');

					$product['id_wp'] = $product_id;
					$this->add_update_product($product);

				} else {

					//product not found, try to create
					
					$product_id = $this->add_create_product($product);
				}
			} else {

				/**
				 * If product had existing Woo ID
				 */
				$this->add_update_product($product);

			}

		}

		$this->send_batch();

		$this->logger->i('Finished product update');

		$this->erp->close_connection();
	}

	/**
	 * Updates a product on Woo based on data from ERP
	 * @param  array 	$product_data	Product data array
	 * @return boolean					true if success false if error
	 */
	public function update_product($product_data) {

		try {

			$data = array (
				'name' => $product_data['name'],				
				'regular_price' => $product_data['price'],
				'stock_quantity' => $product_data['stock']
			);

			$this->woo->put("products/".$product_data['id_wp'],$data);

			return true;

		} catch (Exception $e) {
			$this->output_woo_exception($e);
			return false;
		}

	}

	/**
	 * Adds a product update to the batch on Woo based on data from ERP
	 * @param  array 	$product_data	Product data array
	 */
	public function add_update_product($product_data) {
		// print_r($product_data['categories']); die;
		$data = array (
			'id' => $product_data['id_wp']
		);

		if (UPDATE_NAME && $product_data['name']) {
			$data['name'] = $product_data['name'];
		}

		if (UPDATE_PRICE && $product_data['price']) {
			$data['regular_price'] = $product_data['price'];
		}

		if (UPDATE_STOCK && $product_data['stock']) {
			$data['stock_quantity'] = $product_data['stock'];
			$data['manage_stock'] = true;
		}

		if (UPDATE_IMAGES && $product_data['images']) {
			$data['images'] = array();
			foreach ($product_data['images'] as $image) {
				$data['images'][] = array('src' => $image);
			}
		}

		if (UPDATE_CATEGORIES && $product_data['categories']) {
			foreach ($product_data['categories'] as $category) {
				$category_id = $this->category->get_id($category);
				$data['categories'][] = array('id' => $category_id);
			}
		}

		if (UPDATE_SALE && $product_data['sale']) {
			$data['sale_price'] = $product_data['sale'];
			$data['date_on_sale_from'] = $product_data['sale_start'];
			$data['date_on_sale_to'] = $product_data['sale_end'];
		}

		if ( UPDATE_STATUS ) {
			switch ($product_data['enabled']) {
				case 0:
					$data['status'] = 'draft';
					break;
				
				case 1:
					$data['status'] = 'publish';
					break;

				default:
					# code...
					break;
			}
		}

		$this->add_update($data);

	}

	/**
	 * Gets the Woo ID of a product based on it's SKU
	 * @param  array $product_data The ID or SKU based on the ERP
	 * @return int         The Woo ID or -1 if error
	 */
	public function get_id($product_data) {

		try {
			$product = $this->woo->get('products', array('sku'=>$product_data['sku']));

			if (count($product) < 1) {

				return -1;

			} else {
				return $product[0]->id;
			}

		} catch (HttpClientException $e) {
			$this->output_woo_exception($e);
			return -1;
		}

	}

	/**
	 * Creates a new product on Woo based on the ERP info
	 * @param  array $product_data	the product to create
	 * @return int                	the id of the created product or -1 if error
	 */
	public function create_product($product_data) {

		$data = array (
			'type' => 'simple',
			'sku' => $product_data['sku'],
			'status' => 'draft',
		);

		if (CREATE_NAME && $product_data['name']) {
			$data['name'] = $product_data['name'];
		}

		if (CREATE_PRICE && $product_data['price']) {
			$data['regular_price'] = $product_data['price'];
		}

		if (CREATE_STOCK && $product_data['stock']) {
			$data['stock_quantity'] = $product_data['stock'];
			$data['manage_stock'] = true;
		}

		try {

			$result = $this->woo->post('products', $data);

			if (isset($result->id)) {
				return $result->id;
			} else {
				return -1;
			}

		} catch (HttpClientException $e){
			$this->output_woo_exception($e);
			return -1;
		}

	}

	/**
	 * Ads a creates new product to the batch on Woo based on the ERP info
	 * @param  array $product_data	the product to create
	 */
	public function add_create_product($product_data) {

		$data = array (
			'type' => 'simple',
			'sku' => $product_data['sku'],
			'status' => 'draft',
		);

		if (CREATE_NAME && $product_data['name']) {
			$data['name'] = $product_data['name'];
		}

		if (CREATE_PRICE && $product_data['price']) {
			$data['regular_price'] = $product_data['price'];
		}

		if (CREATE_STOCK && $product_data['stock']) {
			$data['stock_quantity'] = $product_data['stock'];
			$data['manage_stock'] = true;
		}

		if (CREATE_IMAGES && $product_data['images']) {
			$data['images'] = array();
			foreach ($product_data['images'] as $image) {
				$data['images'][] = array('src' => $image);
			}
		}

		if (CREATE_CATEGORIES && $product_data['categories']) {
			foreach ($product_data['categories'] as $category) {
				$category_id = $this->category->get_id($category);
				$data['categories'][] = array('id' => $category_id);
			}
		}

		if (CREATE_SALE && $product_data['sale']) {
			$data['sale_price'] = $product_data['sale'];
			$data['date_on_sale_from'] = $product_data['sale_start'];
			$data['date_on_sale_to'] = $product_data['sale_end'];
		}

		if (CREATE_STATUS) {
			switch ($product_data['enabled']) {
				case 0:
					$data['status'] = 'draft';
					break;
				
				case 1:
					$data['status'] = 'publish';
					break;

				default:
					# code...
					break;
			}
		}

		$this->add_create($data);
	}

	/**
	 * Adds a product to be updated to the batch
	 * @param array $item Product to be updated
	 */
	private function add_update($item) {
		$this->add_call($item, "update");
	}

	/**
	 * Adds a product to be created to the batch
	 * @param array $item Product to be created
	 */
	private function add_create($item) {
		$this->add_call($item, "create");
	}

	/**
	 * Adds a product to the batch
	 * @param array $item Product to be added
	 * @param string $type Type of update (create/update/delete)
	 */
	private function add_call($item, $type) {

		$this->batch[$type][] = $item;

		$this->batch_count++;

		// If batch is full
		if ($this->batch_count > $this->batch_size) {
			$this->send_batch();
		}

	}

	/**
	 * Sends current batch
	 */
	private function send_batch() {

		if ($this->batch_count <= 0)
			return;

		try {

			$this->logger->d("Sending batch ".print_r($this->batch,1));

			$this->process_batch_result( $this->woo->post('products/batch', $this->batch) );

		} catch (HttpClientException $e) {

			$this->output_woo_exception($e);

			$this->logger->e("Error updating batch");

		} finally {

			$this->batch_count = 0;

			$this->batch = array();

		}
	}

	/**
	 * Process the batch results to update IDs and log exceptions.
	 * @param  object $results The results from woo batch update
	 */
	private function process_batch_result($results) {

		if (isset($results->create)) {
			foreach ($results->create as $product) {
				if (!isset($product->error)) {

					/*
						If no errors update wp_id and set status 2 -> updated
					 */

					$this->erp->update_product_id($product->sku, $product->id, '');
					$this->logger->i("Created product ".$product->id);
					$this->erp->set_status($product->id);
				}
				else {
					$this->logger->e("Error creating product ".json_encode($product, JSON_UNESCAPED_SLASHES |    JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT | JSON_PARTIAL_OUTPUT_ON_ERROR | JSON_INVALID_UTF8_SUBSTITUTE));
				}
			}
		}

		if (isset($results->update)) {

			foreach ($results->update as $product) {
				if (!isset($product->error)) {

					$this->erp->set_status($product->id);

				} else {

					$this->logger->e("Error updating product.".json_encode($product, JSON_UNESCAPED_SLASHES |    JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT | JSON_PARTIAL_OUTPUT_ON_ERROR | JSON_INVALID_UTF8_SUBSTITUTE));

					if (isset($product->id)) {
						$this->logger->i("Attempting to reset wp_id ".$product->id);
						$this->erp->reset_wp_id($product->id);
					}

				}
			}
		}

	}

	private function output_woo_exception($e) {
		$this->logger->e(print_r( $e->getMessage(), true ));  // Error message.
	    $this->logger->d(print_r( $e->getRequest(), true ));  // Last request data.
	    $this->logger->d(print_r( $e->getResponse(), true )); // Last response data.
	}

	private function output_woo_data() {

	    // Last request data.
	    $lastRequest = $this->woo->http->getRequest();
	    echo '<pre>Request URL: <code>' . print_r( $lastRequest->getUrl(), true ) . '</code></pre>'; // Requested URL (string).
	    echo '<pre>Method: <code>' . print_r( $lastRequest->getMethod(), true ) . '</code></pre>'; // Request method (string).
	    echo '<pre>Paramters: <code>' . print_r( $lastRequest->getParameters(), true ) . '</code></pre>'; // Request parameters (array).
	    echo '<pre>Headers: <code>' . print_r( $lastRequest->getHeaders(), true ) . '</code></pre>'; // Request headers (array).
	    echo '<pre>Body: <code>' . print_r( $lastRequest->getBody(), true ) . '</code></pre>'; // Request body (JSON).

	    // Last response data.
	    $lastResponse = $this->woo->http->getResponse();
	    echo '<pre>Response Code: <code>' . print_r( $lastResponse->getCode(), true ) . '</code></pre>'; // Response code (int).
	    echo '<pre>Headers: <code>' . print_r( $lastResponse->getHeaders(), true ) . '</code></pre>'; // Response headers (array).
	    echo '<pre>Body: <code>' . print_r( $lastResponse->getBody(), true ) . '</code></pre>'; // Response body (JSON).
	}

}