<?php
/**
* PHP-Logger - Simple PHP logging class with singleton pattern. Log to screen and log file.
* @author  hounw - Thanks to Drew D. Lenhart - snowytech for base logger
* @since   Oct 1, 2020
* @link    https://webifica.com
* @version 0.0.1
*/

defined('DEBUG_LEVEL') or define('DEBUG_LEVEL', '1');


class logger {

    /**
     * Array of severity levels with labels
     * @var [type]
     */
    protected $severities = array(
        '',
        'ERROR',
        'WARNING',
        'INFO',
        'DEBUG'
    );

    /**
    * $log_file - path and log file name
    * @var string
    */
    protected $log_file;

    /**
    * $file - file
    * @var string
    */
    protected $file;

    /**
    * $options - settable options - future use - passed through constructor
    * @var array
    */
    protected $options = array(
        'dateFormat' => "Y-m-d H:i:s",
        'do_echo' => false
    );


    /**
    * Class constructor
    * @param string $log_file - path and filename of log
    * @param array $params
    */
    public function __construct($log_file = 'error.log', $params = array()){
        $this->log_file = $log_file;
        $this->options = array_merge($this->options, $params);
        //Create log file if it doesn't exist.
        if(!file_exists($log_file)){               
            fopen($log_file, 'w') or exit("Can't create $log_file!");
        }
        //Check permissions of file.
        if(!is_writable($log_file)){   
            //throw exception if not writable
            throw new Exception("ERROR: Unable to write to file!", 1);
        }
    }

    /**
    * Info method (write info message)
    * @param string $message
    * @return void
    */
    public function i($message){
        $this->writeLog($message, 3);
    }
    /**
    * Debug method (write debug message)
    * @param string $message
    * @return void
    */
    public function d($message){
        $this->writeLog($message, 4);
    }
    /**
    * Warning method (write warning message)
    * @param string $message
    * @return void
    */
    public function w($message){
        $this->writeLog($message, 2);
    }
    /**
    * Error method (write error message)
    * @param string $message
    * @return void
    */
    public function e($message){
        $this->writeLog($message, 1);
    }
    /**
    * Write to log file
    * @param string $message
    * @param string $severity
    * @return void
    */
    public function writeLog($message, $severity) {

        // grab the url path ( for troubleshooting )
        $path = $_SERVER["SERVER_NAME"] . $_SERVER["REQUEST_URI"];
        //Grab time - based on timezone in php.ini
        $time = date($this->options['dateFormat']);

        if ($this->options['do_echo']) {
            echo "<pre>[$time] [$path] : [".$this->severities[$severity]."] - $message</pre>";
        }

        if (DEBUG_LEVEL >= $severity) {

            // open log file
            if (!is_resource($this->file)) {
                $this->openLog();
            }
            
            // Write time, url, & message to end of file
            fwrite($this->file, "[$time] [$path] : [".$this->severities[$severity]."] - $message" . PHP_EOL);

            if (isset($this->options['db_log'])) {
                $this->options['db_log']->log_event($this->severities[$severity], $message);
            }

        }
        
    }
    /**
    * Open log file
    * @return void
    */
    private function openLog(){
        $openFile = $this->log_file;
        // 'a' option = place pointer at end of file
        $this->file = fopen($openFile, 'a') or exit("Can't open $openFile!");
    }
    /**
     * Class destructor
     */
    public function __destruct(){
        if ($this->file) {
            fclose($this->file);
        }
    }
}